/**
  ********************************************************************************
  *
  * @file    hal_base.h
  * @brief   Header file of Base driver
  *
  * @version V1.0
  * @data    22 Aug 2018
  * @author  AE Team
  * @note
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  ********************************************************************************
  */


#ifndef __HAL_BASE_H__
#define __HAL_BASE_H__

#include <hic.h>
#include "type.h"


/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @addtogroup Base
  * @{
  */

/**
  * @defgroup Base_Public_Types base Public Types
  * @{
  */

/**
  * @brief Clock source select
  */
typedef enum {
	CLOCK_SRC_HRC = 0,
	CLOCK_SRC_LRC = 1,
} clock_src_t;

/**
  * @brief Clock divide select
  */
typedef enum {
	CLOCK_DIV_1   = 0,
	CLOCK_DIV_2   = 1,
	CLOCK_DIV_4   = 2,
	CLOCK_DIV_8   = 3,
	CLOCK_DIV_16  = 4,
	CLOCK_DIV_32  = 5,
	CLOCK_DIV_64  = 6,
	CLOCK_DIV_128 = 7,
} clock_div_t;

/**
  * @brief Reset flag
  */
typedef enum {
	RESET_BORB_FLAG  = (1U << 0),
	RESET_PORB_FLAG  = (1U << 1),
	RESET_PDB_FLAG   = (1U << 2),
	RESET_TOB_FLAG   = (1U << 3),
	RESET_IRSTB_FLAG = (1U << 4),
} reset_flag_t;

/**
  * @defgroup Base_Public_Micros base Public Micros
  * @{
  */
#define GPIO_PC1_EN()		\
do {				\
	PWRCWP  = 0xA5;		\
	SMRSTEN = 0;		\
} while (0)
#define GPIO_MRSTN_EN()		\
do {				\
	PWRCWP  = 0xA5;		\
	SMRSTEN = 1;		\
} while (0)
#define WDT_CLOSE()		\
do {				\
	WDTUL = 0xA5;		\
	WDTC  = 0;		\
	WDTUL = 1;		\
} while (0)

#define CLOCK_UNLOCK()		(OSCWP = 0x55)
#define CLOCK_LOCK()		(OSCWP = 1)
#define ADC_CLOCK_EN()		(ADCCE = 1)
#define ADC_CLOCK_DISABLE()	(ADCCE = 0)
#define T21_CLOCK_EN()		(T21CE = 1)
#define T21_CLOCK_DISABLE()	(T21CE = 0)
#define T20_CLOCK_EN()		(T20CE = 1)
#define T20_CLOCK_DISABLE()	(T20CE = 0)
#define TK_CLOCK_EN()		(TKCE = 1)
#define TK_CLOCK_DISABLE()	(TKCE = 0)
#define I2C_CLOCK_EN()		(IICCE = 1)
#define I2C_CLOCK_DISABLE()	(IICCE = 0)
#define UART0_CLOCK_EN()	(UART0CE = 1)
#define UART0_CLOCK_DISABLE()	(UART0CE = 0)
#define T10_CLOCK_EN()		(T10CE = 1)
#define T10_CLOCK_DISABLE()	(T10CE = 0)

#define sleep()			__Asm IDLE;
#define CLRWDT()   		__Asm CWDT;	
/**
  * @}
  */
	
/**
  * @defgroup Base_Private_Micros base private Micros
  * @{
  */	

#define IS_CLOCK_SRC_TYPE(x)		(((x) == CLOCK_SRC_HRC) || \
					 ((x) == CLOCK_SRC_LRC))
#define IS_CLOCK_DIV_TYPE(x)		(((x) == CLOCK_DIV_1)  || \
					 ((x) == CLOCK_DIV_2)  || \
					 ((x) == CLOCK_DIV_4)  || \
					 ((x) == CLOCK_DIV_8)  || \
					 ((x) == CLOCK_DIV_16) || \
					 ((x) == CLOCK_DIV_32) || \
					 ((x) == CLOCK_DIV_64) || \
					 ((x) == CLOCK_DIV_128))
/**
  * @}
  */

/**
  * @addtogroup Base_Public_Functions
  * @{
  */  
void ram_clear(void);
void delay(uint16_t times);
hal_status_t clock_init(clock_src_t src, clock_div_t div, type_func_t clk_out);
uint32_t get_system_clock(void);
flag_status_t get_reset_flag(reset_flag_t flag);
void clear_reset_flag(reset_flag_t flag);
uint16_t mul_calc(uint8_t mul_a, uint8_t mul_b);
uint32_t div_calc(uint32_t divider, uint16_t divisor, uint16_t *remainder);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#endif /* __HAL_BASE_H */